/*
 * Fabric Management Agent
 *
 * One runs on each Myrinet node and communicates with the fabric
 * management server(s).
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>

#include "libfma.h"
#include "lf_scheduler.h"
#include "lf_channel.h"
#include "lf_fma_flags.h"

#include "fma.h"
#include "fma_fms.h"

/*
 * All global variables live in here
 */
struct fma_vars A;

/*
 * local prototypes
 */
static void fma_init_vars(void);
static int parse_options(int argc, char **argv);
static int fma_load_environment(void);
static int fma_init_process(void);
static void fma_write_pid_file(void);
static int fma_init_process(void);
static void fma_daemonize(void);
static void fma_init_log(void);
static void fma_init_signals(void);
static void fma_startup_msg(void);

static void
usage()
{
  fprintf(stderr, "Usage: fma\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -d - run as daemon\n");
  fprintf(stderr, "  -i - fabric consists of xbars with IDs only\n");
  fprintf(stderr, "  -l <level> - set mapping level\n");
  fprintf(stderr, "  -p - partition mode, require FMS\n");
  fprintf(stderr, "  -x - fabric consists of ID-less xbars only\n");
  fprintf(stderr, "  -D - increment debug level\n");
  fprintf(stderr, "  -s <fms_server> - specify fms server\n");
  fprintf(stderr, "  -R <fms_run> - run directory (for pid and log files)\n");
  fprintf(stderr, "  -L <log_file> - log filename, in <fms_run> if relative\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

int
main(int argc,
     char **argv)
{
  int ms;
  int rc;

  /* initialize libfma */
  lf_init();

  /* initialize error handler */
  lf_set_error_handler(fma_save_error);

  /* initialize all variables to default values */
  fma_init_vars();

  /* first load environmental settings, then command line may override */
  if (fma_load_environment() != 0) LF_ERROR(("loading environment"));
  if (parse_options(argc, argv) != 0) LF_ERROR(("parsing command line"));

  /* Initialize process stuff, signals, stdio, etc. */
  if (fma_init_process() == -1) LF_ERROR(("Error initializing process"));

  /* initialize channels subsystem */
  if (lf_init_channels() == -1) LF_ERROR(("Initializing channels subsystem"));

  /* Initialize Myricom setup */
  sleep(5);			/* give multi-NIC time to come up XXX */
  if (fma_init_myri() == -1) LF_ERROR(("Error initializing Myricom"));

  /* initialize FMS subsystem */
  if (fma_init_fms() == -1) LF_ERROR(("Initializing FMS subsystem"));

  /* Initialize standalone stuff */
  fma_init_standalone();

  /* if we made it this far, we are up and running!  write pid file */
  fma_write_pid_file();

  /* emit a startup message */
  fma_startup_msg();

  /*
   * Just loop handling events forever
   */
  while (!A.done) {
    ms = lf_check_events();	/* look for any scheduled events */

    rc = lf_poll_channels(ms);	/* look for any pending I/O requests */
    if (rc == -1) LF_ERROR(("polling channels"));
  }

  fma_exit(0);

 except:
  fma_perror();
  fma_exit(1);
  return 0;
}

/*
 * Initialize the FMA global data structure
 */
static void
fma_init_vars()
{
  /* initialize FMA variables */
  A.done = FALSE;

  A.xbar_types = FMA_XT_ANY;		/* default is any type of xbar */
  A.map_level = 1;			/* default mapping level */
  A.fma_run = LF_DFLT_FMS_RUN;

  if (gethostname(A.hostname, sizeof(A.hostname)) == -1) {
    LF_ERROR(("Getting local hostname"));
  }

  /* Initialize settings vars */
  fma_init_settings_vars();

  /* vars for the Myricom subsystem */
  if (fma_init_myri_vars() != 0) LF_ERROR(("Initializing Myricom interfaces"));

  /* vars for the FMS subsystem */
  if (fma_init_fms_vars() != 0) LF_ERROR(("Initializing FMS subsystem"));

  /* map information */
  fma_init_map_info_vars();

  /* In case we need to run standalone */
  fma_init_standalone_vars();

  fma_init_tunnel_vars();	/* initialize message tunnel variables */

  return;

 except:
  fma_perror_exit(1);
}

/*
 * Load environment variables
 */
static int
fma_load_environment()
{
  char *s;

  s = getenv(LF_ENV_FMS_SERVER);
  if (s != NULL && *s != '\0') {
    if (fma_set_fms_list(s) != 0) {
      LF_ERROR(("Error getting fms server list from env"));
    }
  }

  s = getenv(LF_ENV_FMS_RUN);
  if (s != NULL && *s != '\0') {
    A.fma_run = s;
  }
  return 0;

 except:
  return -1;
}

/*
 * parse command line args
 */
static int
parse_options(
  int argc,
  char **argv)
{
  extern char *optarg;
  int c;

  /* Set default logfile name */
  A.fma_logfilename = "fma.log";

  while ((c = getopt(argc, argv, "dpL:hDR:s:il:xV")) != EOF) switch (c) {
  case 'd':
    A.fma_background = TRUE;		/* run in background */
    break;
  case 'h':
    usage();
    break;
  case 'L':
    A.fma_logfilename = optarg;
    break;
  case 'p':
    A.fms_required = TRUE;
    break;
  case 'R':
    A.fma_run = optarg;
    break;
  case 's':
    if (fma_set_fms_list(optarg) != 0) {
      LF_ERROR(("Error loading host list from cmd line"));
    }
    break;
  case 'D':
    ++A.debug;		/* increase debug level */
    break;
  case 'i':
    A.xbar_types = FMA_XT_ID_ONLY;
    break;
  case 'l':
    A.map_level = atoi(optarg);
    if (A.map_level < 0 || A.map_level > FMA_MAX_MAPPER_LEVEL) {
      /* XXX fix LF_ERROR!!! */
      fprintf(stderr, "Invalid map level %d, valid range is [0..%d]\n",
	  A.map_level, FMA_MAX_MAPPER_LEVEL);
      exit(1);
    }
    break;
  case 'x':
    A.xbar_types = FMA_XT_NO_IDS;
    break;
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  }

  return 0;

 except:
  return -1;
}

/*
 * perform any cleanup and exit
 */
void
fma_exit(
  int code)
{
  exit(code);
}

/*
 * Initialize stdio/stderr and random number generator
 */
static int
fma_init_process()
{
  /* Setup signals */
  fma_init_signals();

  /* initialize logging */
  fma_init_log();

  /* Run to the background if requested */
  if (A.fma_background) {
    fma_daemonize();
  }

  srandom((unsigned int)time(NULL));

  return 0;
}

/*
 * initialize signals
 * ignore SIGPIPE since we are messing with sockets
 */
void
fma_init_signals()
{
  struct sigaction sa;
  int rc;

  /* Ignore SIGPIPE */
  memset(&sa, 0, sizeof(sa));
  sa.sa_handler = SIG_IGN;
  rc = sigaction(SIGPIPE, &sa, NULL);
  if (rc != 0) {
    perror("Error ignoring SIGPIPE");
    exit(1);
  }
}

/*
 * Write PID file
 */
static void
fma_write_pid_file()
{
  FILE *fp;
  lf_string_t path;

  /* save a PID file */
  sprintf(path, "%s/fma.pid", A.fma_run);
  fp = fopen(path, "w");
  if (fp == NULL) {
    perror(path);
    exit(1);
  }

  /* write our PID to a file */
  fprintf(fp, "%d\n", getpid());
  fclose(fp);
}

/*
 * wrapper for free suitable for use as callback
 */
void
fma_free(
  void *addr)
{
  LF_FREE(addr);
}

/*
 * Redirect output to our log file.  Hijack both stdout and stderr to this file.
 */
static void
fma_init_log()
{
  lf_string_t logfilename;
  FILE *fp;

  /* If going into the background, auto-redirect stdout to our log file */
  if (A.fma_background) {

    /* a logfilename of "-" means leave stdout alone */
    if (strcmp(A.fma_logfilename, "-") != 0) {

      if (A.fma_logfilename[0] == '/') {
	strcpy(logfilename, A.fma_logfilename);
      } else {
	sprintf(logfilename, "%s/%s", A.fma_run, A.fma_logfilename);
      }

      /* say something before redirecting stdout */
      if (isatty(fileno(stdin))) {
	printf("Logging to %s\n", logfilename);
      }

      fp = freopen(logfilename, "w", stdout);
      if (fp == NULL) {
	fprintf(stderr, "Cannot open logfile %s: ", logfilename);
	perror("");
	exit(1);
      }
    }

    /* Now, hijack stderr to stdout */
    dup2(fileno(stdout), fileno(stderr));

    /* No need for stdin */
    close(fileno(stdin));
  }

  /* Set line buffering */
  setvbuf(stdout, (char *)NULL, _IONBF, 0);
  setvbuf(stderr, (char *)NULL, _IONBF, 0);

  return;
}

/*
 * become a background daemon
 */
static void
fma_daemonize()
{ 
  int pid;
  int fd;

  /* Create a new process, so the parent can exit immediately. */
  pid = fork();
  if (pid < 0) {
    LF_ERROR(("Error calling fork()"));

  /* Parent exits successfully. */
  } else if (pid > 0) {
    exit (0);
  }
  
  /* Create a process group, so we won't be killed when our parent
     process group exits. */
  setsid ();		/* always succeeds for child */

  /* Close any left-over file descriptor */
  for (fd=0;fd < 255;fd++) {
    if (fd != fileno(stdout) && fd != fileno(stderr)) {
      close(fd);
    }
  }

#if 0
  /* Write the process ID to a file, taking care that there is
     no other mapper running and holding a lock on this file. */
      
  pid_file = open (filename, O_RDWR | O_CREAT, 0644);
  if (pid_file < 0)
  {
    printf ("could not open %s:", filename);
    perror ("");
    return 0;
  }
  if (lockf (pid_file, F_TLOCK, 0) < 0)
  {
    /* We cannot lock the file, so there must be another mapper
       running. */
    printf ("could not lock %s:", filename);
    perror ("");
    return 0;
  }
  sprintf (buf, "%d\n", (int)getpid ());
  if (write (pid_file, buf, strlen (buf)) == -1)
  {
    printf ("could not write PID to %s:", filename);
    perror ("");
    return 0;
  }
  
  /* Close any open files that might be attached to the controlling
     terminal.  Not doing this is know to prevent the mapper from
     being spawned via ssh. */

  if ((fd = open("/dev/null", O_RDWR, 0)) != -1)
  {
    struct stat s;
    (void)dup2(fd, STDIN_FILENO);
    if (fstat(STDOUT_FILENO, &s) || !S_ISREG(s.st_mode))
      (void)dup2(fd, STDOUT_FILENO);
    if (fstat(STDERR_FILENO, &s) || !S_ISREG(s.st_mode))
      (void)dup2(fd, STDERR_FILENO);
    if (fd > 2)
    {
      (void)close(fd);
    }
  }
#endif
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Print version and some settings info
 */
static void
fma_startup_msg()
{
  fma_log("%s fma starting", Lf_version);

  /* Log some settings */
  if (A.xbar_types == FMA_XT_NO_IDS) {
    fma_log("xbar ID support disabled");
  } else if (A.xbar_types == FMA_XT_ID_ONLY) {
    fma_log("ID-less xbar support disabled");
  }
}
